// Chat-Funktionen
document.addEventListener('DOMContentLoaded', function() {
    console.log("Initialisiere Chat-Funktionen...");
    
    // Chat-Nachrichten laden
    loadChatMessages();
    
    // Event-Listener für das Nachrichtenformular
    const chatForm = document.getElementById('chat-form');
    if (chatForm) {
        chatForm.addEventListener('submit', function(e) {
            e.preventDefault();
            sendMessage();
        });
    } else {
        console.error("Chat-Formular nicht gefunden!");
    }
    
    // Speichern des Nicknames im localStorage
    const nicknameInput = document.getElementById('chat-nickname');
    if (nicknameInput) {
        const savedNickname = localStorage.getItem('chatNickname');
        if (savedNickname) {
            nicknameInput.value = savedNickname;
        }
        
        nicknameInput.addEventListener('change', function() {
            localStorage.setItem('chatNickname', this.value);
        });
    }
});

// Funktion zum Laden der Chat-Nachrichten
function loadChatMessages() {
    const chatMessages = document.getElementById('chat-messages');
    if (!chatMessages) {
        // Versuche den alternativen Container zu finden
        const chatBox = document.getElementById('chat-box');
        if (chatBox) {
            console.log("Verwende 'chat-box' als Container für Nachrichten");
            loadChatMessagesInto(chatBox);
        } else {
            console.error("Chat-Container nicht gefunden!");
        }
        return;
    }
    
    loadChatMessagesInto(chatMessages);
}

// Hilfsfunktion zum Laden der Nachrichten in einen Container
function loadChatMessagesInto(container) {
    // Loading-Nachricht anzeigen
    container.innerHTML = '<div class="text-center p-3"><div class="spinner-border text-primary" role="status"></div><p class="mt-2">Lade Nachrichten...</p></div>';
    
    fetch('/api/chat/messages')
        .then(response => {
            if (!response.ok) {
                throw new Error('Netzwerkantwort war nicht ok: ' + response.status);
            }
            return response.json();
        })
        .then(messages => {
            console.log("Chat-Nachrichten geladen:", messages.length);
            container.innerHTML = '';
            
            if (!Array.isArray(messages) || messages.length === 0) {
                container.innerHTML = '<div class="text-center text-muted py-3">Noch keine Nachrichten vorhanden.</div>';
                return;
            }
            
            // Template für Nachrichten suchen
            const messageTemplate = document.getElementById('chat-message-template');
            
            messages.forEach(msg => {
                if (!msg || typeof msg !== 'object') return;
                
                if (messageTemplate) {
                    // Template-basierte Nachrichtenanzeige
                    const node = document.importNode(messageTemplate.content, true);
                    
                    // Timestamp formatieren
                    if (msg.timestamp) {
                        try {
                            const timestamp = new Date(msg.timestamp);
                            const timeString = timestamp.toLocaleTimeString([], { hour: '2-digit', minute: '2-digit' });
                            const timestampEl = node.querySelector('.message-timestamp');
                            if (timestampEl) {
                                timestampEl.textContent = timeString;
                            }
                        } catch (e) {
                            console.error("Fehler beim Formatieren des Zeitstempels:", e);
                        }
                    }
                    
                    // Nickname setzen
                    const nicknameEl = node.querySelector('.message-nickname');
                    if (nicknameEl) {
                        nicknameEl.textContent = msg.nickname || 'Anonym';
                        nicknameEl.style.color = msg.color || '#3498db';
                    }
                    
                    // Nachricht setzen (als HTML)
                    const messageTextEl = node.querySelector('.message-text');
                    if (messageTextEl) {
                        // Füge Leerzeichen hinzu und setze HTML-Inhalt
                        messageTextEl.innerHTML = ' ' + (msg.message || '');
                    }
                    
                    // Löschen-Button konfigurieren
                    const deleteBtn = node.querySelector('.delete-message-btn');
                    if (deleteBtn) {
                        deleteBtn.setAttribute('data-message-id', msg.id || '');
                        
                        // Löschbutton beim Hover anzeigen
                        const messageDiv = node.querySelector('.chat-message');
                        if (messageDiv) {
                            messageDiv.addEventListener('mouseenter', function() {
                                deleteBtn.classList.remove('opacity-0');
                            });
                            messageDiv.addEventListener('mouseleave', function() {
                                deleteBtn.classList.add('opacity-0');
                            });
                        }
                        
                        // Klick-Event für Löschen
                        deleteBtn.addEventListener('click', function() {
                            const messageId = this.getAttribute('data-message-id');
                            if (!messageId) return;
                            
                            const password = prompt('Bitte geben Sie das Aktionspasswort ein:');
                            if (password) {
                                deleteMessage(messageId, password);
                            }
                        });
                    }
                    
                    container.appendChild(node);
                } else {
                    // Fallback ohne Template
                    const messageDiv = document.createElement('div');
                    messageDiv.className = 'chat-message mb-2';
                    
                    // Timestamp
                    let timeString = '';
                    if (msg.timestamp) {
                        try {
                            const timestamp = new Date(msg.timestamp);
                            timeString = timestamp.toLocaleTimeString([], { hour: '2-digit', minute: '2-digit' });
                        } catch (e) {
                            console.error("Fehler beim Formatieren des Zeitstempels:", e);
                        }
                    }
                    
                    messageDiv.innerHTML = `
                        <span class="small text-muted me-2">${timeString}</span>
                        <strong style="color: ${msg.color || '#3498db'}">${msg.nickname || 'Anonym'}</strong>:
                        <span>${msg.message || ''}</span>
                        <button class="btn btn-sm btn-outline-danger float-end opacity-0" title="Löschen" data-message-id="${msg.id || ''}">
                            <i class="bi bi-x-lg"></i>
                        </button>
                    `;
                    
                    // Löschbutton beim Hover anzeigen
                    const deleteBtn = messageDiv.querySelector('[data-message-id]');
                    if (deleteBtn) {
                        messageDiv.addEventListener('mouseenter', function() {
                            deleteBtn.classList.remove('opacity-0');
                        });
                        messageDiv.addEventListener('mouseleave', function() {
                            deleteBtn.classList.add('opacity-0');
                        });
                        
                        // Klick-Event für Löschen
                        deleteBtn.addEventListener('click', function() {
                            const messageId = this.getAttribute('data-message-id');
                            if (!messageId) return;
                            
                            const password = prompt('Bitte geben Sie das Aktionspasswort ein:');
                            if (password) {
                                deleteMessage(messageId, password);
                            }
                        });
                    }
                    
                    container.appendChild(messageDiv);
                }
            });
            
            // Scroll zum Ende des Chat-Containers
            container.scrollTop = container.scrollHeight;
        })
        .catch(error => {
            console.error('Fehler beim Laden der Chat-Nachrichten:', error);
            container.innerHTML = 
                '<div class="alert alert-danger">Fehler beim Laden der Nachrichten: ' + error.message + '</div>' +
                '<div class="text-center mt-3"><button class="btn btn-primary" onclick="loadChatMessages()">Erneut versuchen</button></div>';
        });
}

// Funktion zum Senden einer Nachricht
function sendMessage() {
    const nicknameInput = document.getElementById('chat-nickname') || document.getElementById('nickname');
    const messageInput = document.getElementById('chat-message') || document.getElementById('message-input');
    const passwordInput = document.getElementById('chat-password') || document.getElementById('password-chat');
    
    if (!nicknameInput || !messageInput || !passwordInput) {
        console.error("Chat-Formularelemente nicht gefunden!");
        alert("Formularelemente nicht gefunden. Bitte Seite neu laden.");
        return;
    }
    
    const nickname = nicknameInput.value.trim();
    const message = messageInput.value.trim();
    const password = passwordInput.value;
    
    if (!nickname) {
        alert('Bitte geben Sie einen Nickname ein');
        nicknameInput.focus();
        return;
    }
    
    if (!message) {
        alert('Bitte geben Sie eine Nachricht ein');
        messageInput.focus();
        return;
    }
    
    if (!password) {
        alert('Bitte geben Sie das Aktionspasswort ein');
        passwordInput.focus();
        return;
    }
    
    // Status anzeigen, falls vorhanden
    const statusEl = document.getElementById('chat-status');
    if (statusEl) {
        statusEl.innerHTML = '<span class="text-info">Sende Nachricht...</span>';
    }
    
    // Button deaktivieren während des Sendens
    const submitButton = document.querySelector('#chat-form button[type="submit"]');
    let originalText = '';
    if (submitButton) {
        originalText = submitButton.innerHTML;
        submitButton.disabled = true;
        submitButton.innerHTML = '<span class="spinner-border spinner-border-sm" role="status" aria-hidden="true"></span> Senden...';
    }
    
    // Nachricht per AJAX senden
    fetch('/api/chat/messages', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
        },
        body: JSON.stringify({
            nickname: nickname,
            message: message,
            password: password
        })
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            // Nachrichteneingabefeld leeren und Nachrichten neu laden
            messageInput.value = '';
            loadChatMessages();
            
            // Status aktualisieren
            if (statusEl) {
                statusEl.innerHTML = '<span class="text-success">Nachricht gesendet!</span>';
                setTimeout(() => {
                    statusEl.innerHTML = '';
                }, 3000);
            }
        } else {
            if (statusEl) {
                statusEl.innerHTML = '<span class="text-danger">Fehler: ' + (data.error || 'Unbekannter Fehler') + '</span>';
            } else {
                alert('Fehler: ' + (data.error || 'Unbekannter Fehler'));
            }
        }
    })
    .catch(error => {
        console.error('Fehler beim Senden der Nachricht:', error);
        if (statusEl) {
            statusEl.innerHTML = '<span class="text-danger">Fehler: ' + error.message + '</span>';
        } else {
            alert('Fehler beim Senden der Nachricht: ' + error.message);
        }
    })
    .finally(() => {
        // Button wieder aktivieren
        if (submitButton) {
            submitButton.disabled = false;
            submitButton.innerHTML = originalText;
        }
    });
}

// Funktion zum Löschen einer Nachricht
function deleteMessage(messageId, password) {
    if (!messageId || !password) return;
    
    fetch('/api/chat/delete/' + messageId, {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
        },
        body: JSON.stringify({
            password: password
        })
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            loadChatMessages();
        } else {
            alert('Fehler: ' + (data.error || 'Unbekannter Fehler'));
        }
    })
    .catch(error => {
        console.error('Fehler beim Löschen der Nachricht:', error);
        alert('Fehler beim Löschen der Nachricht: ' + error.message);
    });
}