// Dateivorschau-Funktionen mit erweitertem Debugging
console.log("file_preview.js wird geladen...");

document.addEventListener('DOMContentLoaded', function() {
    console.log("DOM geladen, initialisiere Dateivorschau-Funktionen...");
    
    // Überprüfe, ob Bootstrap verfügbar ist
    console.log("Bootstrap verfügbar:", typeof bootstrap !== 'undefined');
    console.log("Bootstrap Modal verfügbar:", typeof bootstrap !== 'undefined' && typeof bootstrap.Modal !== 'undefined');
    
    // Überprüfe, ob das Modal-Element existiert
    const modalElement = document.getElementById('filePreviewModal');
    console.log("Modal-Element gefunden:", modalElement !== null);
    
    // Event-Listener für alle Vorschau-Links
    const previewLinks = document.querySelectorAll('.file-preview-link');
    console.log("Vorschau-Links gefunden:", previewLinks.length);
    
    if (previewLinks.length > 0) {
        console.log("Beispiel-Link data-filename:", previewLinks[0].getAttribute('data-filename'));
        
        // Teste, ob die Links korrekt funktionieren
        previewLinks.forEach((link, index) => {
            console.log(`Link ${index}: data-filename="${link.getAttribute('data-filename')}"`);
            link.addEventListener('click', function(e) {
                console.log(`Link ${index} wurde geklickt`);
                e.preventDefault();
                const filename = this.getAttribute('data-filename');
                console.log("Dateivorschau angefordert für:", filename);
                openFilePreview(filename);
            });
        });
        
        console.log("Event-Listener für Vorschau-Links hinzugefügt");
    } else {
        console.warn("Keine Vorschau-Links gefunden! Überprüfen Sie, ob Ihre Links die Klasse 'file-preview-link' haben.");
    }

    // Event-Delegation für dynamisch hinzugefügte Vorschau-Links
    document.addEventListener('click', function(e) {
        if (e.target && (e.target.classList.contains('file-preview-link') || 
                        (e.target.parentElement && e.target.parentElement.classList.contains('file-preview-link')))) {
            console.log("Link über Event-Delegation geklickt");
            e.preventDefault();
            
            // Bestimme das korrekte Element (Link oder Kind-Element)
            const linkElement = e.target.classList.contains('file-preview-link') ? 
                               e.target : e.target.parentElement;
            
            const filename = linkElement.getAttribute('data-filename');
            console.log("Dateivorschau angefordert für:", filename);
            openFilePreview(filename);
        }
    });
    
    // Füge einen Test-Button hinzu, um das Modal direkt zu testen
    if (window.location.hostname === 'localhost' || window.location.hostname === '127.0.0.1') {
        const testButton = document.createElement('button');
    testButton.textContent = "Test Modal";
    testButton.className = "btn btn-info position-fixed bottom-0 end-0 m-3";
    testButton.style.zIndex = "9999";
    testButton.onclick = function() {
        console.log("Test-Button geklickt");
        if (modalElement) {
            try {
                const modal = new bootstrap.Modal(modalElement);
                modal.show();
                console.log("Modal wurde durch Test-Button geöffnet");
            } catch (error) {
                console.error("Fehler beim Öffnen des Modals:", error);
                alert("Fehler beim Öffnen des Modals: " + error.message);
            }
        } else {
            console.error("Modal-Element nicht gefunden!");
            alert("Modal-Element nicht gefunden! Überprüfen Sie, ob das Modal-HTML eingefügt wurde.");
        }
    };
    document.body.appendChild(testButton);
}});

// Funktion zum Öffnen der Dateivorschau
function openFilePreview(filename) {
    console.log("openFilePreview aufgerufen für:", filename);
    if (!filename) {
        console.error("Kein Dateiname angegeben!");
        return;
    }
    
    // AJAX-Anfrage, um Dateiinformationen zu erhalten
    console.log("Sende API-Anfrage an:", '/preview-api/' + encodeURIComponent(filename));
    
    fetch('/preview-api/' + encodeURIComponent(filename))
        .then(response => {
            console.log("API-Antwort erhalten, Status:", response.status);
            if (!response.ok) {
                throw new Error('Fehler bei der API-Anfrage: ' + response.status);
            }
            return response.json();
        })
        .then(data => {
            console.log("API-Daten erhalten:", data);
            
            if (!data.success) {
                throw new Error(data.error || 'Unbekannter Fehler');
            }
            
            const modalElement = document.getElementById('filePreviewModal');
            if (!modalElement) {
                console.error("Modal-Element nicht gefunden!");
                alert("Modal-Element nicht gefunden! Stellen Sie sicher, dass das Modal-HTML in die Seite eingefügt wurde.");
                return;
            }
            
            // Modale Elemente aktualisieren
            console.log("Modal-Elemente werden aktualisiert...");
            
            const filenameElement = document.getElementById('preview-filename');
            if (filenameElement) {
                filenameElement.textContent = data.filename;
            } else {
                console.error("Element 'preview-filename' nicht gefunden!");
            }
            
            const filetypeElement = document.getElementById('preview-filetype');
            if (filetypeElement) {
                filetypeElement.textContent = data.mime_type;
            } else {
                console.error("Element 'preview-filetype' nicht gefunden!");
            }
            
            const filesizeElement = document.getElementById('preview-filesize');
            if (filesizeElement) {
                filesizeElement.textContent = data.size;
            } else {
                console.error("Element 'preview-filesize' nicht gefunden!");
            }
            
            const modifiedElement = document.getElementById('preview-modified');
            if (modifiedElement) {
                modifiedElement.textContent = data.modified;
            } else {
                console.error("Element 'preview-modified' nicht gefunden!");
            }
            
            // Alle Vorschaubereiche ausblenden
            console.log("Blende alle Vorschaubereiche aus...");
            
            const imagePrevElement = document.getElementById('image-preview');
            const pdfPrevElement = document.getElementById('pdf-preview');
            const videoPrevElement = document.getElementById('video-preview');
            const audioPrevElement = document.getElementById('audio-preview');
            const genericPrevElement = document.getElementById('generic-preview');
            
            if (imagePrevElement) imagePrevElement.classList.add('d-none');
            if (pdfPrevElement) pdfPrevElement.classList.add('d-none');
            if (videoPrevElement) videoPrevElement.classList.add('d-none');
            if (audioPrevElement) audioPrevElement.classList.add('d-none');
            if (genericPrevElement) genericPrevElement.classList.add('d-none');
            
            // Download-Button aktualisieren
            console.log("Aktualisiere Download-Button...");
            
            const downloadBtn = document.getElementById('modal-download-btn');
            if (downloadBtn) {
                downloadBtn.href = data.url;
                downloadBtn.setAttribute('download', data.filename);
            } else {
                console.error("Element 'modal-download-btn' nicht gefunden!");
            }
            
            // Je nach Dateityp den entsprechenden Vorschaubereich anzeigen
            console.log("Zeige Vorschaubereich für Dateityp:", 
                        data.is_image ? "Bild" : 
                        data.is_pdf ? "PDF" : 
                        data.is_video ? "Video" : 
                        data.is_audio ? "Audio" : "Generisch");
            
            if (data.is_image) {
                if (imagePrevElement) {
                    const previewImage = document.getElementById('preview-image');
                    if (previewImage) {
                        previewImage.src = data.url;
                        imagePrevElement.classList.remove('d-none');
                    } else {
                        console.error("Element 'preview-image' nicht gefunden!");
                    }
                }
            } 
            else if (data.is_pdf) {
                if (pdfPrevElement) {
                    const previewPdf = document.getElementById('preview-pdf');
                    if (previewPdf) {
                        previewPdf.src = data.url;
                        pdfPrevElement.classList.remove('d-none');
                    } else {
                        console.error("Element 'preview-pdf' nicht gefunden!");
                    }
                }
            } 
            else if (data.is_video) {
                if (videoPrevElement) {
                    const previewVideo = document.getElementById('preview-video');
                    if (previewVideo) {
                        const videoSource = previewVideo.querySelector('source');
                        if (videoSource) {
                            videoSource.src = data.url;
                            videoSource.type = data.mime_type;
                            previewVideo.load();
                            videoPrevElement.classList.remove('d-none');
                        } else {
                            console.error("Video-Source-Element nicht gefunden!");
                        }
                    } else {
                        console.error("Element 'preview-video' nicht gefunden!");
                    }
                }
            } 
            else if (data.is_audio) {
                if (audioPrevElement) {
                    const previewAudio = document.getElementById('preview-audio');
                    if (previewAudio) {
                        const audioSource = previewAudio.querySelector('source');
                        if (audioSource) {
                            audioSource.src = data.url;
                            audioSource.type = data.mime_type;
                            previewAudio.load();
                            audioPrevElement.classList.remove('d-none');
                        } else {
                            console.error("Audio-Source-Element nicht gefunden!");
                        }
                    } else {
                        console.error("Element 'preview-audio' nicht gefunden!");
                    }
                }
            } 
            else {
                if (genericPrevElement) {
                    const previewDownload = document.getElementById('preview-download');
                    if (previewDownload) {
                        previewDownload.href = data.url;
                        previewDownload.setAttribute('download', data.filename);
                        genericPrevElement.classList.remove('d-none');
                    } else {
                        console.error("Element 'preview-download' nicht gefunden!");
                    }
                }
            }
            
            // Modal öffnen
            console.log("Versuche Modal zu öffnen...");
            
            try {
                const filePreviewModal = new bootstrap.Modal(modalElement);
                filePreviewModal.show();
                console.log("Modal wurde geöffnet");
            } catch (error) {
                console.error("Fehler beim Öffnen des Modals:", error);
                alert("Fehler beim Öffnen des Modals: " + error.message);
            }
        })
        .catch(error => {
            console.error("Fehler bei der Dateivorschau:", error);
            alert('Fehler beim Laden der Dateivorschau: ' + error.message);
        });
}

// Event-Handler für das Schließen des Modals hinzufügen
document.addEventListener('DOMContentLoaded', function() {
    const modalElement = document.getElementById('filePreviewModal');
    if (modalElement) {
        modalElement.addEventListener('hidden.bs.modal', function() {
            // Entferne alle Modal-Backdrops manuell
            const backdrops = document.querySelectorAll('.modal-backdrop');
            backdrops.forEach(backdrop => backdrop.remove());
            
            // Entferne die modal-open Klasse vom body
            document.body.classList.remove('modal-open');
            document.body.style.overflow = '';
            document.body.style.paddingRight = '';
        });
    }
});

// Alternative Funktionalität: Direkte Navigation zur Dateivorschau-Seite
function openFullPreviewPage(filename) {
    console.log("Öffne Vollbild-Vorschau für:", filename);
    window.location.href = '/file-preview/' + encodeURIComponent(filename);
}