// static/js/script.js - Version 16 - Modern & Fix für Upload
// Mit erweiterten Funktionen und besserem UI-Feedback

document.addEventListener('DOMContentLoaded', () => {
    console.log("--- 🚀 Modern Script.js DOM loaded (v16) ---");

    // --- Passwort aus data-Attribut lesen ---
    // Der einfachste und zuverlässigste Weg, das Passwort global zu speichern
    const SHARED_PASSWORD = '0911'; // Hardcoded für robuste Funktionalität
    console.debug("SHARED_PASSWORD set for script usage");

    // --- Gemeinsame Funktionen ---
    const showStatus = (elementId, message, isError = false, timeout = 5000) => {
        const statusElement = document.getElementById(elementId);
        if (statusElement) {
            const textClass = isError ? 'text-danger' : 'text-success';
            const currentMessage = message;
            
            // Moderne Statusanzeige mit Icons
            const iconHTML = isError 
                ? '<i class="bi bi-exclamation-circle me-2"></i>' 
                : '<i class="bi bi-check-circle me-2"></i>';
                
            statusElement.innerHTML = message ? `${iconHTML}${currentMessage}` : '';
            statusElement.className = `mt-2 form-text ${currentMessage ? textClass : ''}`;
            
            if (timeout > 0 && currentMessage) {
                 const currentTimeoutMessage = currentMessage;
                 setTimeout(() => {
                     if (statusElement.textContent.includes(currentTimeoutMessage)) {
                         statusElement.innerHTML = '';
                         statusElement.className = 'mt-2 form-text';
                     }
                 }, timeout);
            } else if (!currentMessage) {
                 statusElement.innerHTML = '';
                 statusElement.className = 'mt-2 form-text';
            }
        } else {
            console.warn(`Status element #${elementId} not found.`);
        }
    };
    console.debug("Helper function 'showStatus' defined.");

    // --- Dateiliste (Index & Verwaltung) ---
    try {
        console.debug("Setting up File List View...");
        const fileListContainer = document.getElementById('file-list') || document.getElementById('admin-file-list');
        const viewToggleButtons = document.querySelectorAll('.view-toggle');
        const refreshButton = document.getElementById('refresh-button') || document.getElementById('refresh-button-admin');
        
        if (viewToggleButtons.length > 0 && fileListContainer) {
            viewToggleButtons.forEach(button => { 
                button.addEventListener('click', () => { 
                    const v = button.dataset.view; 
                    
                    // Smooth transition for view change
                    fileListContainer.style.opacity = '0';
                    fileListContainer.style.transition = 'opacity 0.3s ease';
                    
                    setTimeout(() => {
                        fileListContainer.className = fileListContainer.className.replace(/grid-view|list-view/g,'').trim() + ` ${v}-view`; 
                        viewToggleButtons.forEach(b => b.classList.remove('active')); 
                        button.classList.add('active'); 
                        localStorage.setItem('fileViewPreference', v);
                        
                        fileListContainer.style.opacity = '1';
                    }, 300);
                }); 
            });
            
            const savedView = localStorage.getItem('fileViewPreference') || 'grid'; 
            fileListContainer.className = fileListContainer.className.replace(/grid-view|list-view/g,'').trim() + ` ${savedView}-view`; 
            viewToggleButtons.forEach(b => {
                b.classList.remove('active'); 
                if(b.dataset.view === savedView) b.classList.add('active');
            }); 
            console.debug("File List View toggles OK.");
        } else { 
            console.warn("File List View elements not found or incomplete."); 
        }
        
        if (refreshButton) { 
            refreshButton.addEventListener('click', () => {
                // Animiertes Refresh-Icon
                refreshButton.querySelector('i').classList.add('rotate-animation');
                setTimeout(() => window.location.reload(), 500);
            }); 
            console.debug("Refresh button listener OK."); 
        } else { 
            console.warn("Refresh button not found."); 
        }

        // Funktion zum Kopieren des Links in die Zwischenablage
        function copyLinkToClipboard(link) {
            navigator.clipboard.writeText(link)
                .then(() => {
                    displayFlashMessage('Link in die Zwischenablage kopiert!', 'success');
                })
                .catch(err => {
                    console.error('Fehler beim Kopieren des Links:', err);
                    displayFlashMessage('Fehler beim Kopieren des Links!', 'danger');
                });
        }

        document.body.addEventListener('click', async (event) => {
            const cb = event.target.closest('.copy-link-btn');
            if (cb && !cb.disabled) {
                // Animation hinzufügen
                const originalHTML = cb.innerHTML;
                cb.innerHTML = '<i class="bi bi-check"></i> Kopiert!';
                cb.classList.add('btn-success');
                cb.classList.remove('btn-outline-secondary');
                
                copyLinkToClipboard(cb.dataset.link);
                
                setTimeout(() => {
                    cb.innerHTML = originalHTML;
                    cb.classList.remove('btn-success');
                    cb.classList.add('btn-outline-secondary');
                }, 2000);
            }
        });

        console.debug("Copy link listener OK.");
    } catch(e) { 
        console.error("Error setting up File List View:", e); 
    }

    // --- Dateivorschau Modal für ältere Preview-Trigger --- 
    try {
        console.debug("Setting up Preview Modal (Backward compatibility)...");
        const previewModalElement = document.getElementById('previewModal');
        if (previewModalElement) {
            console.debug("Preview modal element #previewModal found.");
            const previewModalBody = document.getElementById('previewModalBody'); 
            const previewModalLabel = document.getElementById('previewModalLabel');
            
            if (!previewModalBody || !previewModalLabel) { 
                console.error("Preview modal body or label element NOT found!"); 
            } else {
                console.debug("Preview modal body and label found.");
                const loadingHtml = `
                    <div class="d-flex justify-content-center align-items-center h-100 p-5">
                        <div class="spinner-border text-primary" role="status" style="width:3rem;height:3rem;">
                            <span class="visually-hidden">Lade...</span>
                        </div>
                    </div>`;
                let currentMediaElement = null;
                
                previewModalElement.addEventListener('show.bs.modal', function (event) {
                    const button = event.relatedTarget; 
                    if (!button) return; 
                    
                    const fileUrl = button.dataset.fileUrl; 
                    const fileName = button.dataset.fileName; 
                    
                    if (!fileUrl || !fileName) {
                        console.error("Preview trigger missing data"); 
                        previewModalBody.innerHTML = '<p class="text-danger p-4">Fehler: Fehlende Infos.</p>';
                        return;
                    } 
                    
                    const isImage = button.dataset.isImage === 'true'; 
                    const isPdf = button.dataset.isPdf === 'true'; 
                    const isVideo = button.dataset.isVideo === 'true'; 
                    const isAudio = button.dataset.isAudio === 'true'; 
                    const fileSize = button.closest('.file-item')?.querySelector('.file-size small')?.textContent || '';
                    
                    console.debug(`Previewing: ${fileName}`); 
                    previewModalLabel.textContent = `Vorschau: ${fileName}`; 
                    previewModalBody.innerHTML = loadingHtml; 
                    currentMediaElement = null;
                    
                    setTimeout(() => { 
                        let element = null; 
                        try { 
                            if (isImage) {
                                element = document.createElement('img');
                                element.src = fileUrl;
                                element.alt = `Vorschau`;
                                element.className = 'preview-media preview-image';
                                element.onerror = () => {
                                    previewModalBody.innerHTML = `<p class="text-danger p-4">Bild nicht ladbar.</p>`;
                                };
                                element.loading = 'lazy';
                            } else if (isPdf) {
                                element = document.createElement('iframe');
                                element.src = fileUrl;
                                element.className = 'preview-media preview-pdf';
                                element.title = "PDF Vorschau";
                            } else if (isVideo) {
                                element = document.createElement('video');
                                element.controls = true;
                                element.preload = "metadata";
                                element.className = 'preview-media preview-video';
                                const s = document.createElement('source');
                                s.src = fileUrl;
                                const e = fileName.split('.').pop().toLowerCase();
                                if (['mp4','webm','ogg'].includes(e)) s.type = `video/${e}`;
                                element.appendChild(s);
                                element.insertAdjacentHTML('beforeend','<p>Format nicht unterstützt.</p>');
                                currentMediaElement = element;
                            } else if (isAudio) {
                                element = document.createElement('audio');
                                element.controls = true;
                                element.preload = "metadata";
                                element.style.width = '100%';
                                const s = document.createElement('source');
                                s.src = fileUrl;
                                const e = fileName.split('.').pop().toLowerCase();
                                if (['mp3','wav','ogg'].includes(e)) s.type = `audio/${e==='mp3'?'mpeg':e}`;
                                element.appendChild(s);
                                element.insertAdjacentHTML('beforeend','<p>Format nicht unterstützt.</p>');
                                currentMediaElement = element;
                            } 
                            
                            if (element) {
                                if (!element.onerror || element.tagName !== 'IMG') {
                                    previewModalBody.innerHTML = '';
                                    previewModalBody.appendChild(element);
                                    if (isPdf) {
                                        previewModalBody.insertAdjacentHTML('beforeend','<p class="mt-2 small text-muted">PDF erfordert Browser-Unterstützung.</p>');
                                    }
                                }
                            } else {
                                previewModalBody.innerHTML = `
                                    <div class="p-4 text-center">
                                        <i class="bi bi-file-earmark fs-1 text-muted mb-3 d-block"></i>
                                        <p>Keine Vorschau für "${fileName}" verfügbar.</p>
                                        <a href="${fileUrl}" class="btn btn-primary mt-2" download="${fileName}">
                                            <i class="bi bi-download"></i> Herunterladen ${fileSize?`(${fileSize})`:''}</a>
                                    </div>`;
                            }
                        } catch (e) { 
                            console.error("Error creating preview content:", e); 
                            previewModalBody.innerHTML = '<p class="text-danger p-4">Fehler beim Erstellen der Vorschau.</p>'; 
                        }
                    }, 100);
                });
                console.debug("'show.bs.modal' listener added for preview.");
                
                previewModalElement.addEventListener('hidden.bs.modal', function () { 
                    console.debug("Preview Modal 'hidden.bs.modal' triggered."); 
                    if (currentMediaElement) {
                        if (typeof currentMediaElement.pause === 'function') currentMediaElement.pause();
                        currentMediaElement.removeAttribute('src');
                        if (typeof currentMediaElement.load === 'function') currentMediaElement.load();
                    } 
                    const iframe = previewModalBody.querySelector('iframe');
                    if (iframe) {
                        iframe.src = 'about:blank';
                    } 
                    previewModalBody.innerHTML = loadingHtml; 
                    previewModalLabel.textContent = 'Dateivorschau'; 
                    currentMediaElement = null; 
                });
                console.debug("'hidden.bs.modal' listener added for preview.");
            }
        } else { 
            console.warn("Preview modal element #previewModal NOT found."); 
        }
        console.debug("Preview Modal setup finished.");
    } catch(e) { 
        console.error("Error setting up Preview Modal:", e); 
    }

    // --- Datei Upload --- (KORRIGIERT)
    try {
        console.debug("Looking for Upload form...");
        const uploadForm = document.getElementById('upload-form');
        if (uploadForm) {
            console.debug("Upload form element #upload-form found.");
            const fileInput = document.getElementById('file-input'); 
            const dropZone = document.getElementById('drop-zone'); 
            const uploadStatus = document.getElementById('upload-status'); 
            const passwordUploadInput = document.getElementById('password-upload');
            const uploadSubmitButton = uploadForm.querySelector('button[type="submit"]');
            
            if (!passwordUploadInput || !fileInput) {
                console.error("Upload form inputs missing!");
            } else {
                // WICHTIG: preventDefault hier, damit das Formular nicht direkt submitted wird
                uploadForm.addEventListener('submit', function(e) {
                    e.preventDefault();
                    console.log(">>> Upload form SUBMIT event triggered and prevented! <<<");
                    
                    // Handler wird im Code unten implementiert
                    handleUploadSubmit();
                });
                
                // Funktion zum Handling des Uploads
                const handleUploadSubmit = async () => {
                    console.log(">>> Handle Upload Submit called <<<");
                    
                    // Prüfe, ob direkt über fileInput oder via Drag & Drop Dateien ausgewählt wurden
                    let uploadFiles = null;
                    
                    if (fileInput && fileInput.files && fileInput.files.length > 0) {
                        console.log("Using files from file input element");
                        uploadFiles = fileInput.files;
                    } else if (window.droppedFiles && window.droppedFiles.length > 0) {
                        console.log("Using dropped files");
                        uploadFiles = window.droppedFiles;
                    }
                    
                    // Wenn keine Dateien gefunden wurden
                    if (!uploadFiles || uploadFiles.length === 0) {
                        showStatus('upload-status', 'Bitte Datei(en) auswählen!', true);
                        console.log("No files selected.");
                        return;
                    }
                
                    // Passwort direkt verwenden
                    const passwordInput = passwordUploadInput.value.trim();
                    
                    // Einfacher Passwort-Check
                    if (passwordInput !== SHARED_PASSWORD) {
                        showStatus('upload-status', 'Falsches Aktionspasswort!', true);
                        console.log("Password Check FAILED.");
                        return;
                    }
                
                    console.debug("Password Check OK. Proceeding with upload...");
                    
                    // UI-Feedback beim Upload
                    if (uploadSubmitButton) {
                        uploadSubmitButton.disabled = true;
                        uploadSubmitButton.innerHTML = '<span class="spinner-border spinner-border-sm me-2" role="status" aria-hidden="true"></span> Wird hochgeladen...';
                    }
                    
                    // Moderne Statusanzeige
                    showStatus('upload-status', 'Dateien werden hochgeladen...', false, 0);
                    
                    // Elegantes Highlight während des Uploads
                    if (dropZone) {
                        dropZone.innerHTML = `
                            <div class="text-center">
                                <div class="spinner-border text-primary mb-3" role="status"></div>
                                <p class="text-primary">Upload läuft...</p>
                            </div>`;
                    }
                    
                    // Erstelle ein neues FormData-Objekt
                    const formData = new FormData();
                    
                    // Füge das Passwort zum FormData hinzu
                    formData.append('password', passwordInput);
                    
                    // Füge alle Dateien zum FormData hinzu
                    for (let i = 0; i < uploadFiles.length; i++) {
                        formData.append('file', uploadFiles[i]);
                    }
                    
                    try {
                        const response = await fetch('/upload', {
                            method: 'POST',
                            body: formData
                        });
                        
                        console.log("Upload response status:", response.status);
                        
                        let data;
                        try {
                            data = await response.json();
                            console.log("Upload response data:", data);
                        } catch (error) {
                            console.error("Error parsing JSON:", error);
                            data = { success: false, error: "Ungültige Serverantwort" };
                        }
                        
                        if (response.ok && data.success) {
                            // Erfolgreiche Anzeige
                            showStatus('upload-status', data.message || 'Datei(en) erfolgreich hochgeladen!', false);
                            
                            // Erfolgsanimation im Drop-Zone
                            if (dropZone) {
                                dropZone.innerHTML = `
                                    <div class="text-center">
                                        <i class="bi bi-check-circle-fill text-success display-4 mb-2"></i>
                                        <p class="text-success">Upload erfolgreich abgeschlossen!</p>
                                    </div>`;
                            }
                            
                            // Erfolgsanimation für Button
                            if (uploadSubmitButton) {
                                uploadSubmitButton.disabled = false;
                                uploadSubmitButton.innerHTML = '<i class="bi bi-check-circle"></i> Hochgeladen!';
                                uploadSubmitButton.classList.remove('btn-primary');
                                uploadSubmitButton.classList.add('btn-success');
                            }
                            
                            // Seite nach kurzer Verzögerung neu laden
                            setTimeout(() => window.location.reload(), 1500);
                        } else {
                            const errorMessage = data.error || `Serverfehler (Status: ${response.status})`;
                            showStatus('upload-status', errorMessage, true);
                            
                            // Fehleranzeige im Drop-Zone
                            if (dropZone) {
                                dropZone.innerHTML = `
                                    <div class="text-center">
                                        <i class="bi bi-exclamation-circle-fill text-danger display-4 mb-2"></i>
                                        <p class="text-danger">Upload fehlgeschlagen</p>
                                        <p class="text-muted small">Bitte versuchen Sie es erneut</p>
                                    </div>`;
                            }
                            
                            console.error("Upload fail:", errorMessage);
                        }
                    } catch (error) {
                        console.error("Upload Fetch Error:", error);
                        showStatus('upload-status', 'Netzwerkfehler beim Upload.', true);
                        
                        // Fehleranzeige im Drop-Zone
                        if (dropZone) {
                            dropZone.innerHTML = `
                                <div class="text-center">
                                    <i class="bi bi-wifi-off text-danger display-4 mb-2"></i>
                                    <p class="text-danger">Netzwerkfehler</p>
                                    <p class="text-muted small">Bitte überprüfen Sie Ihre Verbindung</p>
                                </div>`;
                        }
                    } finally {
                        // UI zurücksetzen (außer bei Erfolg)
                        if (uploadSubmitButton && !uploadSubmitButton.classList.contains('btn-success')) {
                            uploadSubmitButton.disabled = false;
                            uploadSubmitButton.innerHTML = '<i class="bi bi-upload"></i> Hochladen';
                        }
                        
                        // Zurücksetzen der gespeicherten Dateien
                        window.droppedFiles = null;
                    }
                };
                
                // Drag & Drop Funktionalität mit modernen Effekten
                if (dropZone) {
                    console.debug("Setting up Drop Zone...");
                    ['dragenter', 'dragover', 'dragleave', 'drop'].forEach(eventName => {
                        dropZone.addEventListener(eventName, preventDefaults, false);
                        document.body.addEventListener(eventName, preventDefaults, false);
                    });

                    ['dragenter', 'dragover'].forEach(eventName => {
                        dropZone.addEventListener(eventName, highlight, false);
                    });

                    ['dragleave', 'drop'].forEach(eventName => {
                        dropZone.addEventListener(eventName, unhighlight, false);
                    });

                    dropZone.addEventListener('drop', handleDrop, false);

                    function preventDefaults(e) {
                        e.preventDefault();
                        e.stopPropagation();
                    }

                    function highlight() {
                        dropZone.classList.add('border-primary');
                        dropZone.classList.add('bg-light-blue');
                        // Animation für das Icon
                        const icon = dropZone.querySelector('.bi');
                        if (icon) {
                            icon.classList.add('text-primary');
                            icon.classList.add('bounce-animation');
                        }
                    }

                    function unhighlight() {
                        dropZone.classList.remove('border-primary');
                        dropZone.classList.remove('bg-light-blue');
                        // Animation entfernen
                        const icon = dropZone.querySelector('.bi');
                        if (icon) {
                            icon.classList.remove('text-primary');
                            icon.classList.remove('bounce-animation');
                        }
                    }

                    function handleDrop(e) {
                        const dt = e.dataTransfer;
                        const files = dt.files;
                        
                        // Dateilisten-Anzeige mit Animation
                        if (files.length > 0) {
                            // Erstelle einen temporären FormData-Speicher für die Dateien
                            window.droppedFiles = files;
                            
                            // Visuelles Feedback
                            const fileList = Array.from(files).map(file => {
                                // Dateigröße formatieren
                                let size = file.size;
                                let sizeDisplay = "";
                                if (size < 1024) {
                                    sizeDisplay = size + " B";
                                } else if (size < 1024 * 1024) {
                                    sizeDisplay = (size / 1024).toFixed(1) + " KB";
                                } else {
                                    sizeDisplay = (size / (1024 * 1024)).toFixed(1) + " MB";
                                }
                                
                                // Icon je nach Dateityp
                                let iconClass = "bi-file";
                                if (file.type.startsWith("image/")) {
                                    iconClass = "bi-file-image";
                                } else if (file.type.startsWith("video/")) {
                                    iconClass = "bi-file-play";
                                } else if (file.type.startsWith("audio/")) {
                                    iconClass = "bi-file-music";
                                } else if (file.type.includes("pdf")) {
                                    iconClass = "bi-file-pdf";
                                } else if (file.type.includes("zip") || file.type.includes("rar")) {
                                    iconClass = "bi-file-zip";
                                }
                                
                                return `
                                    <div class="file-item-preview d-flex align-items-center mb-2 p-2 border rounded fade-in">
                                        <i class="bi ${iconClass} me-2 text-primary"></i>
                                        <div class="flex-grow-1 text-truncate">${file.name}</div>
                                        <span class="ms-2 text-muted small">${sizeDisplay}</span>
                                    </div>`;
                            }).join('');
                            
                            dropZone.innerHTML = `
                                <div class="p-2">
                                    <p class="text-success mb-3"><i class="bi bi-check-circle-fill me-2"></i>${files.length} Dateien ausgewählt</p>
                                    <div class="file-list-preview">
                                        ${fileList}
                                    </div>
                                </div>`;
                        }
                    }
                    console.debug("Drop Zone setup finished.");
                }
            }
        } else {
            console.debug("Upload form not found on this page.");
        }
        console.debug("Upload form setup finished.");
    } catch(e) { 
        console.error("Error setting up Upload form:", e); 
    }

    // --- Datei löschen (Verwaltung) ---
    try {
        console.debug("Setting up Delete button listener...");
        document.body.addEventListener('click', async (event) => { 
            const deleteBtn = event.target.closest('.delete-file-btn'); 
            if (deleteBtn) { 
                event.preventDefault(); 
                const filename = deleteBtn.dataset.filename; 
                const fileItem = deleteBtn.closest('.file-item'); 
                
                if (!filename) return; 
                
                // Modernes Löschen-Dialog mit SweetAlert wenn verfügbar, sonst Standard-Confirm
                if (typeof Swal !== 'undefined') {
                    Swal.fire({
                        title: 'Datei löschen?',
                        text: `Möchten Sie "${filename}" wirklich löschen?`,
                        icon: 'warning',
                        showCancelButton: true,
                        confirmButtonColor: '#dc3545',
                        cancelButtonColor: '#6c757d',
                        confirmButtonText: 'Ja, löschen!',
                        cancelButtonText: 'Abbrechen'
                    }).then((result) => {
                        if (result.isConfirmed) {
                            deleteFile(filename, fileItem);
                        }
                    });
                } else if (confirm(`Datei "${filename}" wirklich löschen?`)) {
                    deleteFile(filename, fileItem);
                }
            }
        });
        
        // Refaktorierte Löschfunktion
        async function deleteFile(filename, fileItem) {
            try { 
                // Visuelles Feedback beim Löschen
                if (fileItem) {
                    fileItem.style.opacity = '0.5';
                    fileItem.style.pointerEvents = 'none';
                    
                    // Animation hinzufügen
                    fileItem.style.transition = 'all 0.3s ease';
                    fileItem.classList.add('deleting');
                }
                
                const response = await fetch(`/delete_file/${encodeURIComponent(filename)}`, {
                    method: 'POST'
                }); 
                
                if (response.redirected && response.url.includes('/login')) {
                    alert('Sitzung abgelaufen.');
                    window.location.href = '/login';
                    return;
                } 
                
                if (response.ok) {
                    const result = await response.json().catch(() => ({success: true})); 
                    
                    if (result.success) {
                        if (fileItem) {
                            // Animation beim Entfernen
                            fileItem.style.height = '0';
                            fileItem.style.margin = '0';
                            fileItem.style.padding = '0';
                            fileItem.style.overflow = 'hidden';
                            
                            setTimeout(() => {
                                fileItem.remove();
                            }, 300);
                        }
                        displayFlashMessage(`Datei "${filename}" gelöscht.`, 'success');
                    } else {
                        alert(`Fehler: ${result.error || 'Unbekannter Serverfehler'}`);
                        // Visuelles Feedback zurücksetzen
                        if (fileItem) {
                            fileItem.style.opacity = '1';
                            fileItem.style.pointerEvents = 'auto';
                            fileItem.classList.remove('deleting');
                        }
                    }
                } else if (response.status === 403) {
                    alert('Keine Berechtigung.');
                    window.location.href = '/login';
                } else {
                    const res = await response.json().catch(() => ({}));
                    alert(`Fehler: ${res.error || `Status ${response.status}`}`);
                    
                    // Visuelles Feedback zurücksetzen
                    if (fileItem) {
                        fileItem.style.opacity = '1';
                        fileItem.style.pointerEvents = 'auto';
                        fileItem.classList.remove('deleting');
                    }
                }
            } catch (error) {
                console.error("Delete Error:", error);
                alert('Netzwerkfehler.');
                
                // Visuelles Feedback zurücksetzen
                if (fileItem) {
                    fileItem.style.opacity = '1';
                    fileItem.style.pointerEvents = 'auto';
                    fileItem.classList.remove('deleting');
                }
            }
        }
        
        console.debug("Delete button listener setup finished.");
    } catch(e) { 
        console.error("Error setting up Delete listener:", e); 
    }

    // --- Chat ---
    try {
        console.debug("Setting up Chat...");
        const chatForm = document.getElementById('chat-form');
        const chatBox = document.getElementById('chat-box');
        const chatMessageTemplate = document.getElementById('chat-message-template');
        const chatLoadingMsg = document.getElementById('chat-loading-msg');

        // Definiere Funktionen im Scope, falls Chat-Elemente existieren
        if (chatForm && chatBox && chatMessageTemplate) {
            console.debug("Chat elements found. Defining functions and listeners.");
            const msgInput = document.getElementById('message-input');
            const nickInput = document.getElementById('nickname');
            const pwChatInput = document.getElementById('password-chat');
            const chatStatus = document.getElementById('chat-status');

            const displayChatMessage = (msg) => {
                console.log("displayChatMessage called with:", msg);
                if (!chatMessageTemplate || !chatBox) {
                    console.log("chatMessageTemplate or chatBox not found.");
                    return;
                }
                
                const c = chatMessageTemplate.content.cloneNode(true);
                const ts = c.querySelector('.message-timestamp');
                const ns = c.querySelector('.message-nickname');
                const t = c.querySelector('.message-text');
                const db = c.querySelector('.delete-message-btn');
                try {
                    ts.textContent = new Date(msg.timestamp).toLocaleTimeString([], {
                        hour: '2-digit',
                        minute: '2-digit'
                    });
                } catch (e) {
                    console.error("Error formatting timestamp:", e); // Hinzugefügt
                }
                ns.textContent = msg.nickname || 'Unbekannt';
                ns.style.color = msg.color || '#333';
                t.innerHTML = msg.message || '';
                db.dataset.messageId = msg.id || '';
                chatBox.appendChild(c);
            };

            const loadChatMessages = async () => { // Entferne window.
                console.debug("loadChatMessages called");
                console.log("chatBox:", chatBox); // Hinzugefügt
                if (!chatBox) {
                    console.log("Chat box not found.");
                    return;
                }
                const sb = chatBox.scrollHeight - chatBox.clientHeight <= chatBox.scrollTop + 15;
                if (chatLoadingMsg) chatLoadingMsg.style.display = 'block';
                try {
                    console.log("Versuche, die API aufzurufen..."); // Hinzugefügt
                    const r = await fetch('/api/chat/messages');
                    console.log("API response:", r); // Hinzugefügt
                    if (!r.ok) {
                        console.error("API response not OK:", r); // Hinzugefügt
                        throw new Error(`HTTP ${r.status}`);
                    }
                    const ms = await r.json();
                    console.log("Chat messages:", ms); // Hinzugefügt
                    if (!ms || !Array.isArray(ms)) { // Hinzugefügt
                        console.error("Chat messages is not a valid array:", ms); // Hinzugefügt
                    }
                    chatBox.innerHTML = '';
                    if (ms.length === 0) {
                        chatBox.innerHTML = '<p class="text-center text-muted p-3">Noch keine Nachrichten.</p>';
                    } else {
                        ms.forEach(displayChatMessage);
                    }
                    if (sb) {
                        chatBox.scrollTop = chatBox.scrollHeight;
                    }
                } catch (err) {
                    console.error('Chat Load Error:', err);
                    chatBox.innerHTML = `<p class="text-danger text-center p-3">Fehler beim Laden.</p>`;
                } finally {
                    if (chatLoadingMsg) chatLoadingMsg.style.display = 'none';
                }
            };

            chatForm.addEventListener('submit', async (e) => {
                e.preventDefault();
                const nn = nickInput.value.trim();
                const msg = msgInput.value.trim();
                const pw = pwChatInput.value;
                if (!pw || pw !== SHARED_PASSWORD) {
                    showStatus('chat-status', 'Falsches Aktionspasswort!', true);
                    return;
                }
                if (!nn || !msg) {
                    showStatus('chat-status', 'Nickname/Nachricht fehlt!', true);
                    return;
                }
                showStatus('chat-status', 'Sende...', false, 0);
                try {
                    const r = await fetch('/api/chat/messages', {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/json'
                        },
                        body: JSON.stringify({
                            nickname: nn,
                            message: msg,
                            password: pw
                        })
                    });
                    const s = await r.json();
                    if (r.ok && s.success) {
                        msgInput.value = '';
                        showStatus('chat-status', '');
                        displayChatMessage(s.message);
                        chatBox.scrollTop = chatBox.scrollHeight;
                    } else {
                        showStatus('chat-status', `Fehler: ${s.error||'Serverfehler'}`, true);
                    }
                } catch (err) {
                    console.error("Chat Send Error:", err);
                    showStatus('chat-status', 'Netzwerkfehler.', true);
                }
            });
            if (chatBox) {
                chatBox.addEventListener('click', async (event) => {
                    const b = event.target.closest('.delete-message-btn');
                    if (!b) return;
                    const i = b.dataset.messageId;
                    const l = b.closest('.chat-message');
                    if (!i) return;
                    const p = pwChatInput.value;
                    if (!p || p !== SHARED_PASSWORD) {
                        alert('Passwort im Feld oben eingeben.');
                        return;
                    }
                    if (confirm('Nachricht löschen?')) {
                        try {
                            const r = await fetch(`/api/chat/delete/${i}`, {
                                method: 'POST',
                                headers: {
                                    'Content-Type': 'application/json'
                                },
                                body: JSON.stringify({
                                    password: p
                                })
                            });
                            const s = await r.json();
                            if (r.ok && s.success) {
                                if (l) l.remove();
                            } else {
                                alert(`Fehler: ${s.error||'Serverfehler'}`);
                            }
                        } catch (err) {
                            console.error("Chat Delete Error:", err);
                            alert('Netzwerkfehler.');
                        }
                    }
                });
            }
            console.debug("Chat listeners added.");

            // Rufe loadChatMessages auf, sobald die Chat-Elemente vorhanden sind
            console.log("Calling loadChatMessages directly from script.js");
            loadChatMessages(); // Direktaufruf hier
        } else {
            console.debug("Chat elements not found on this page."); // OK für andere Seiten
        }
        console.debug("Chat setup finished.");
    } catch (e) { // Fange spezifische Fehler im Chat-Setup ab
        console.error("Error setting up Chat:", e);
    }

    // --- Kalender-JS ist in kalender.html ---

    console.log("--- Script.js DOMContentLoaded finished ---");

}); // Ende DOMContentLoaded

// Flash Message Helper
function displayFlashMessage(message, category = 'info') {
    const c = document.getElementById('flash-message-container');
    if (c) {
        const a = document.createElement('div');
        a.className = `alert alert-${category} alert-dismissible fade show shadow-sm mb-4`;
        a.setAttribute('role', 'alert');
        a.innerHTML = `${message}<button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>`;
        c.appendChild(a);
    } else {
        console.warn("Flash message container not found.");
    }
}